# -*- coding: UTF-8 -*-
# Uklonjen from __future__ import absolute_import
################################    Titulky.com #################################

import os

# Python 3 uvozi umjesto six.moves
from urllib.request import HTTPCookieProcessor, build_opener, install_opener, Request, urlopen
from urllib.parse import urlencode
from http import cookiejar as http_cookiejar # Preimenovano radi lakše zamjene

import time
import calendar
import re
from ..utilities import languageTranslate, log, getFileSize

from ..seeker import SubtitlesDownloadError, SubtitlesErrors


def search_subtitles(file_original_path, title, tvshow, year, season, episode, set_temp, rar, lang1, lang2, lang3, stack): #standard input
    # need to filter titles like <Localized movie name> (<Movie name>)
    br_index = title.find('(')
    if br_index > -1:
        title = title[:br_index]
    title = title.strip()
    session_id = "0"
    client = TitulkyClient()
    subtitles_list = client.search_subtitles(file_original_path, title, tvshow, year, season, episode, set_temp, rar, 'Czech', 'Slovak', 'EN')
    return subtitles_list, session_id, ""  #standard output


def download_subtitles(subtitles_list, pos, zip_subs, tmp_sub_dir, sub_folder, session_id): #standard input

    subtitle_id = subtitles_list[pos]['ID']
    client = TitulkyClient()
    # NAPOMENA: settings_provider je zadržan, mora biti definiran u izvršnom okruženju
    username = settings_provider.getSetting("Titulkyuser")
    password = settings_provider.getSetting("Titulkypass")
    if password == '' or username == '':
        log(__name__, 'Credentials to Titulky.com not provided')
    else:
        if client.login(username, password) == False:
            log(__name__, 'Login to Titulky.com failed. Check your username/password at the addon configuration')
            raise SubtitlesDownloadError(SubtitlesErrors.INVALID_CREDENTIALS_ERROR,
                                         "Login to Titulky.com failed. Check your username/password at the addon configuration")
            return True, subtitles_list[pos]['language_name'], ""
        log(__name__, 'Login successfull')
        
    log(__name__, f'Get page with subtitle (id={subtitle_id})')
    content = client.get_subtitle_page(subtitle_id) # content je bytes
    
    control_img = client.get_control_image(content) # radi s bytes (content)
    if control_img is not None: # is not None (Py3 stil)
        log(__name__, 'Found control image :(, asking user for input')
        # subtitle limit was reached .. we need to ask user to rewrite image code :(
        log(__name__, 'Download control image')
        img = client.get_file(control_img) # img je bytes
        
        # Korištenje 'with' za sigurnije rukovanje datotekama
        img_file_path = os.path.join(tmp_sub_dir, 'image.png')
        with open(img_file_path, 'wb') as img_file:
            img_file.write(img)
            # flush i close su implicitni s 'with', ali zadržavamo ih radi sigurnosti/stila
            img_file.flush()
        # img_file.close() # Zatvoreno automatski s 'with'

        # NAPOMENA: captcha_cb je zadržan, mora biti definiran u izvršnom okruženju
        solution = captcha_cb(img_file_path) 
        if solution:
            log(__name__, f'Solution provided: {solution}')
            content = client.get_subtitle_page2(content, solution, subtitle_id)
            control_img2 = client.get_control_image(content)
            if control_img2 is not None: # is not None (Py3 stil)
                log(__name__, 'Invalid control text')
                raise SubtitlesDownloadError(SubtitlesErrors.CAPTCHA_RETYPE_ERROR, "Invalid control text")
                #xbmc.executebuiltin("XBMC.Notification(%s,%s,1000,%s)" % (__scriptname__,"Invalid control text",os.path.join(__cwd__,'icon.png')))
                return True, subtitles_list[pos]['language_name'], ""
            else:
                log(__name__, 'Dialog was canceled')
                log(__name__, 'Control text not confirmed, returning in error')
                return True, subtitles_list[pos]['language_name'], ""

    wait_time = client.get_waittime(content)
    cannot_download = client.get_cannot_download_error(content)
    if cannot_download is not None: # is not None (Py3 stil)
        log(__name__, 'Subtitles cannot be downloaded, user needs to login')
        raise SubtitlesDownloadError(SubtitlesErrors.NO_CREDENTIALS_ERROR, "Subtitles cannot be downloaded, user needs to login")
        return True, subtitles_list[pos]['language_name'], ""
        
    link = client.get_link(content)
    log(__name__, 'Got the link, wait %i seconds before download' % (wait_time))
    delay = wait_time
    
    # NAPOMENA: delay_cb je zadržan, mora biti definiran u izvršnom okruženju
    if 'delay_cb' in globals():
        delay_cb(wait_time + 2)
    else:
        for i in range(wait_time + 1):
            line2 = 'Download will start in %i seconds' % (delay,)
            #xbmc.executebuiltin("XBMC.Notification(%s,%s,1000,%s)" % (__scriptname__,line2,os.path.join(__cwd__,'icon.png')))
            delay -= 1
            time.sleep(1)

    log(__name__, 'Downloading subtitle zip')
    data = client.get_file(link) # data je bytes
    log(__name__, f'Saving to file {zip_subs}')
    
    # Korištenje 'with' za sigurnije rukovanje datotekama
    with open(zip_subs, 'wb') as zip_file:
        zip_file.write(data)
    # zip_file.close() # Zatvoreno automatski s 'with'
    
    return True, subtitles_list[pos]['language_name'], "zip" #standard output


def lang_titulky2xbmclang(lang):
    if lang == 'CZ':
        return 'Czech'
    if lang == 'SK':
        return 'Slovak'
    return 'English'


def lang_xbmclang2titulky(lang):
    if lang == 'Czech':
        return 'CZ'
    if lang == 'Slovak':
        return 'SK'
    return 'EN'


def get_episode_season(episode, season):
    # Korištenje f-stringova i fiksirano formatiranje
    return f'S{get2DigitStr(int(season))}E{get2DigitStr(int(episode))}'


def get2DigitStr(number):
    # Korištenje f-stringova i fiksirano formatiranje
    return f'{int(number):02d}'


def lang2_opensubtitles(lang):
    lang = lang_titulky2xbmclang(lang)
    return languageTranslate(lang, 0, 2)


class TitulkyClient(object):

    def __init__(self):
        self.cookies = {}
        self.server_url = 'https://www.titulky.com'
        opener = build_opener(HTTPCookieProcessor(http_cookiejar.LWPCookieJar()))
        opener.addheaders = [('User-agent', 'Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 ( .NET CLR 3.5.30729)')]
        install_opener(opener)

    def login(self, username, password):
        log(__name__, 'Logging in to Titulky.com')
        # Kodiranje POST podataka u bajtove (potrebno za Request)
        login_postdata = urlencode({'Login': username, 'Password': password, 'foreverlog': '1', 'Detail2': ''}).encode('utf-8')
        request = Request(self.server_url + '/index.php', login_postdata)
        response = urlopen(request)
        log(__name__, 'Got response')
        
        # response.read() je bytes, provjeravamo s bytes literalom
        if response.read().find(b'BadLogin') > -1:
            return False

        log(__name__, 'Storing Cookies')
        self.cookies = {}
        # Regex se izvršava na stringu headera
        set_cookie_header = response.headers.get('Set-Cookie')
        if set_cookie_header:
            try:
                self.cookies['CRC'] = re.search(r'CRC=(\S+);', set_cookie_header, re.IGNORECASE | re.DOTALL).group(1)
                self.cookies['LogonLogin'] = re.search(r'LogonLogin=(\S+);', set_cookie_header, re.IGNORECASE | re.DOTALL).group(1)
                self.cookies['LogonId'] = re.search(r'LogonId=(\S+);', set_cookie_header, re.IGNORECASE | re.DOTALL).group(1)
            except AttributeError:
                # Nismo uspjeli pronaći sve potrebne kolačiće
                log(__name__, 'Failed to parse Logon cookies from header.')
                return False
        else:
             log(__name__, 'No Set-Cookie header found after login.')
             return False # Neuspjeh u prijavi
        
        return True

    def search_subtitles(self, file_original_path, title, tvshow, year, season, episode, set_temp, rar, lang1, lang2, lang3):
        url = self.server_url + '/index.php?' + urlencode({'Fulltext': title, 'FindUser': ''})
        if not (tvshow is None or tvshow == ''): # is None umjesto == None
            title2 = tvshow + ' ' + get_episode_season(episode, season)
            url = self.server_url + '/index.php?' + urlencode({'Fulltext': title2, 'FindUser': ''})
            
        req = Request(url)
        try:
            size = getFileSize(file_original_path)
            # Korištenje f-stringa (kao u originalnom kodu)
            file_size = f'{float(size) / (1024 * 1024):.2f}'
        except:
            file_size = '-1'
            
        log(__name__, f'Opening {url}')
        response = urlopen(req)
        content = response.read() # content je bytes
        response.close()
        log(__name__, 'Done')
        subtitles_list = []
        max_downloads = 1
        log(__name__, 'Searching for subtitles')
        
        # Regex na bytes sadržaju. Podaci se moraju dekodirati prilikom spremanja.
        # Fiksiranje Regex literala na bytes (b'')
        for row in re.finditer(b'<tr class=\"r(.+?)</tr>', content, re.IGNORECASE | re.DOTALL):
            item = {}
            log(__name__, 'New subtitle found')
            try:
                # Svi group() rezultati su bytes, moraju se dekodirati u stringove za daljnju obradu.
                
                # ID
                item['ID'] = row.group(1)
                item['ID'] = re.search(rb'[^\<]+<td[^\<]+<a href=\"[\w-]+-(?P<data>\d+).htm\"', item['ID'], re.IGNORECASE | re.DOTALL).group('data').decode('utf-8')
                
                # Title
                item['title'] = row.group(1)
                item['title'] = re.search(rb'[^\<]+<td[^\<]+<a[^\>]+>(<div[^\>]+>)?(?P<data>[^\<]+)', item['title'], re.IGNORECASE | re.DOTALL).group('data').decode('utf-8')
                
                # Sync
                item['sync'] = ''
                sync_found = re.search(rb'((.+?)</td>)[^\>]+>[^\<]*<a(.+?)title=\"(?P<data>[^\"]+)', row.group(1), re.IGNORECASE | re.DOTALL)
                if sync_found:
                    item['sync'] = sync_found.group('data').decode('utf-8')
                    
                # TVShow
                item['tvshow'] = re.search(rb'((.+?)</td>){2}[^\>]+>(?P<data>[^\<]+)', row.group(1), re.IGNORECASE | re.DOTALL).group('data').decode('utf-8')
                
                # Year
                item['year'] = re.search(rb'((.+?)</td>){3}[^\>]+>(?P<data>[^\<]+)', row.group(1), re.IGNORECASE | re.DOTALL).group('data').decode('utf-8')
                
                # Downloads
                item['downloads'] = re.search(rb'((.+?)</td>){4}[^\>]+>(?P<data>[^\<]+)', row.group(1), re.IGNORECASE | re.DOTALL).group('data').decode('utf-8')
                
                # Language
                item['lang'] = re.search(rb'((.+?)</td>){5}[^\>]+><img alt=\"(?P<data>\w{2})\"', row.group(1), re.IGNORECASE | re.DOTALL).group('data').decode('utf-8')
                
                # Number of Discs
                item['numberOfDiscs'] = re.search(rb'((.+?)</td>){6}[^\>]+>(?P<data>[^\<]+)', row.group(1), re.IGNORECASE | re.DOTALL).group('data').decode('utf-8')
                
                # Size
                item['size'] = re.search(rb'((.+?)</td>){7}[^\>]+>(?P<data>[\d\.]+)', row.group(1), re.IGNORECASE | re.DOTALL).group('data').decode('utf-8')
                
            except Exception as e:
                log(__name__, f'Exception when parsing subtitle, all I got is  {str(item)}')
                # log(__name__, f'Error details: {e}') # Opcionalno za debug
                continue
                
            if item['sync'] == '': # if no sync info is found, just use title instead of None
                item['filename'] = item['title']
            else:
                item['filename'] = item['sync']
            item['language_flag'] = f"flags/{lang2_opensubtitles(item['lang'])}.gif"

            sync = False
            # Fiksirano: provjera ako je item['sync'] prazan string
            if item['sync'] and file_original_path.find(item['sync']) > -1: 
                log(__name__, 'found sync : filename match')
                sync = True
            if file_size == item['size']:
                log(__name__, 'found sync : size match')
                sync = True
            item['sync'] = sync

            try:
                downloads = int(item['downloads'])
                if downloads > max_downloads:
                    max_downloads = downloads
            except:
                downloads = 0
            item['downloads'] = downloads

            if year:
                if not item['year'] == year:
                    log(__name__, f'year does not match, ignoring {str(item)}')
                    continue
            
            lang = lang_titulky2xbmclang(item['lang'])

            item['language_name'] = lang
            item['mediaType'] = 'mediaType'
            item['rating'] = '0'

            if lang in [lang1, lang2, lang3]:
                subtitles_list.append(item)
            else:
                log(__name__, f'language does not match, ignoring {str(item)}')
                
        # computing ratings is based on downloads
        for subtitle in subtitles_list:
             # Fiksirano: osigurava da dijeljenje s nulom ne dogodi
            divisor = max_downloads if max_downloads > 0 else 1
            subtitle['rating'] = str(int(subtitle['downloads'] * 10 / divisor))
            
        return subtitles_list

    def get_cannot_download_error(self, content):
        # content je bytes, provjeravamo s bytes literalom
        if content.find(b'CHYBA') > -1:
            return True

    def get_waittime(self, content):
        # content je bytes, regex je bytes
        for matches in re.finditer(b'CountDown\((\d+)\)', content, re.IGNORECASE | re.DOTALL):
            # Rezultat se dekodira
            return int(matches.group(1).decode('utf-8'))

    def get_link(self, content):
        # content je bytes, regex je bytes
        for matches in re.finditer(b'<a.+id=\"downlink\" href="([^\"]+)\"', content, re.IGNORECASE | re.DOTALL):
            # Rezultat se dekodira
            return str(matches.group(1).decode('utf-8'))

    def get_control_image(self, content):
        # content je bytes, regex je bytes
        for matches in re.finditer(b'\.\/(captcha\/captcha\.php)', content, re.IGNORECASE | re.DOTALL):
            # Rezultat se dekodira
            return '/' + str(matches.group(1).decode('utf-8'))
        return None

    def get_file(self, link):
        url = self.server_url + link
        log(__name__, f'Downloading file {url}')
        req = Request(url)
        req = self.add_cookies_into_header(req)
        response = urlopen(req)
        
        set_cookie_header = response.headers.get('Set-Cookie')
        if set_cookie_header:
            # Regex se izvršava na stringu headera
            phpsessid = re.search(r'PHPSESSID=(\S+);', set_cookie_header, re.IGNORECASE | re.DOTALL)
            if phpsessid:
                log(__name__, "Storing PHPSessionID")
                self.cookies['PHPSESSID'] = phpsessid.group(1)
                
        content = response.read() # content je bytes
        log(__name__, 'Done')
        response.close()
        return content

    def get_subtitle_page2(self, content, code, id):
        url = self.server_url + '/idown.php'
        post_data = {'downkod': code, 'titulky': id, 'zip': 'z', 'securedown': '2', 'histstamp': ''}
        # Kodiranje POST podataka u bajtove
        req = Request(url, urlencode(post_data).encode('utf-8'))
        req = self.add_cookies_into_header(req)
        log(__name__, f'Opening {url} POST:{str(post_data)}')
        response = urlopen(req)
        content = response.read() # content je bytes
        log(__name__, 'Done')
        response.close()
        return content

    def get_subtitle_page(self, id):
        timestamp = str(calendar.timegm(time.gmtime()))
        url = self.server_url + '/idown.php?' + urlencode({'R': timestamp, 'titulky': id, 'histstamp': '', 'zip': 'z'})
        log(__name__, f'Opening {url}')
        req = Request(url)
        req = self.add_cookies_into_header(req)
        response = urlopen(req)
        content = response.read() # content je bytes
        log(__name__, 'Done')
        response.close()
        return content

    def add_cookies_into_header(self, request):
        cookies_string = ""
        cookies_list = []
        try:
            # Korištenje f-stringova za čitljivost
            if 'LogonLogin' in self.cookies:
                 cookies_list.append(f"LogonLogin={self.cookies['LogonLogin']}")
            if 'LogonId' in self.cookies:
                 cookies_list.append(f"LogonId={self.cookies['LogonId']}")
            if 'CRC' in self.cookies:
                 cookies_list.append(f"CRC={self.cookies['CRC']}")
        except KeyError:
            pass
            
        cookies_string = "; ".join(cookies_list)

        if 'PHPSESSID' in self.cookies:
            if cookies_string:
                cookies_string += "; "
            cookies_string += "PHPSESSID=" + self.cookies['PHPSESSID']
            
        request.add_header('Cookie', cookies_string)
        return request
        
        